//
//  ColorSelectionViewController.swift
//  VisaSensoryBrandingDemo
//

import Foundation
import UIKit
import SnapKit
import VisaSensoryBranding

class ColorSelectionViewController: UIViewController {
    
    let containerHeight = 80
    let containerPadding = 20
    let scrollView = UIScrollView()
    let stackView = UIStackView()
    var activeField: UITextField?
    let colorCodeTextField = UITextField()
    
    var pickerViews: [UILabel] = []
    var customColorBlock: UIView = UIView()
    
    override func viewDidLoad() {
        
        navigationItem.title = "Branding Color Selection"
        let okBtn = UIBarButtonItem(title: "OK",
                                    style: .plain,
                                    target: self,
                                    action: #selector(okBtnItemClicked))
        okBtn.tintColor = .white
        navigationItem.rightBarButtonItem = okBtn        
        navigationController?.navigationBar.tintColor = UIColor.white

        view.backgroundColor = .white
        
        view.addSubview(scrollView)
        scrollView.snp.makeConstraints { make in
            make.edges.equalTo(view.safeAreaLayoutGuide)
        }
        
        scrollView.addSubview(stackView)
        
        stackView.axis = .vertical
        stackView.alignment = .fill
        stackView.distribution = .equalSpacing
        stackView.spacing = 0
        stackView.snp.makeConstraints { make in
            make.width.equalTo(scrollView.snp.width).offset(containerPadding * -2)
            make.leading.equalTo(scrollView.snp.leading).offset(containerPadding)
            make.trailing.equalTo(scrollView.snp.trailing)
            make.top.equalTo(scrollView.snp.top)
            make.bottom.equalTo(scrollView.snp.bottom)
        }
        
        _ = addThemeItem(id: 0,
                     color: .white,
                     title: MainViewController.whiteTheme.title,
                     desc: MainViewController.whiteTheme.desc)
        
        _ = addThemeItem(id: 1,
                     color: .visaBlue,
                     title: MainViewController.blueTheme.title,
                     desc: MainViewController.blueTheme.desc)
        
        customColorBlock = addThemeItem(id: 2,
                     color: .darkGray,
                     title: MainViewController.customThemeWithHint.title,
                     desc: MainViewController.customThemeWithHint.desc)
        
        let customContainer = UIView()
        stackView.addArrangedSubview(customContainer)
        customContainer.snp.makeConstraints { maker in
            maker.height.equalTo(containerHeight)
            maker.leading.equalTo(stackView.snp.leading)//.offset(containerPadding)
            maker.trailing.equalTo(stackView.snp.trailing)//.offset(containerPadding * -1)
        }
        let hashTag = UILabel()
        hashTag.text = "#"
        hashTag.textAlignment = .right
        customContainer.addSubview(hashTag)
        hashTag.snp.makeConstraints { make in
            make.width.equalTo(64)
            make.leading.equalTo(customContainer)
        }
        
        customContainer.addSubview(colorCodeTextField)
        colorCodeTextField.textAlignment = .left
        colorCodeTextField.autocapitalizationType = .allCharacters
        colorCodeTextField.layer.borderWidth = 1.0
        colorCodeTextField.snp.makeConstraints { make in
            make.width.equalTo(120)
            make.leading.equalTo(hashTag.snp.trailing)
        }

        colorCodeTextField.delegate = self
        colorCodeTextField.text = "666666"
        colorCodeTextField.textColor = .black
        
        switch MainViewController.params.backgroundColor {
        case UIColor.white:
            pickerViews[0].text = "✓"
        case UIColor.visaBlue:
            pickerViews[1].text = "✓"
        default:
            pickerViews[2].text = "✓"
            let currCustomColor = MainViewController.params.backgroundColor.hexString
            colorCodeTextField.text = String(currCustomColor!.dropFirst())
            customColorBlock.backgroundColor = MainViewController.params.backgroundColor
        }
    }
    
    func addThemeItem(id: Int,
                      color: UIColor,
                      title: String,
                      desc: String) -> UIView {
        
        let themeContainer = UIView()
        let themeColorBlock = UIView()
        let themeTitle = UILabel()
        let themeDesc = UILabel()
        let themePickedFlag = UILabel()
        
        stackView.addArrangedSubview(themeContainer)
        themeContainer.snp.makeConstraints { maker in
            maker.height.equalTo(containerHeight)
            maker.leading.equalTo(stackView.snp.leading)//.offset(containerPadding)
            maker.trailing.equalTo(stackView.snp.trailing)//.offset(containerPadding * -1)
        }
        
        themeColorBlock.backgroundColor = color
        themeColorBlock.layer.borderWidth = 1
        themeColorBlock.layer.borderColor = UIColor.darkGray.cgColor
        themeContainer.addSubview(themeColorBlock)
        themeColorBlock.snp.makeConstraints { maker in
            maker.width.equalTo(40)
            maker.height.equalTo(40)
            maker.leading.equalTo(themeContainer)
            maker.centerY.equalTo(themeContainer.snp.centerY)
        }
        
        themeTitle.text = title
        themeTitle.textColor = .black
        themeTitle.font = .systemFont(ofSize: 14)
        themeTitle.isUserInteractionEnabled = false
        themeContainer.addSubview(themeTitle)
        themeTitle.snp.makeConstraints { maker in
            maker.height.equalTo(32)
            maker.top.equalTo(themeColorBlock.snp.top).offset(-4)
            maker.leading.equalTo(themeColorBlock.snp.trailing).offset(20)
            maker.trailing.equalTo(themeContainer)
        }
        
        
        themeDesc.text = desc
        themeDesc.textColor = .lightGray
        themeDesc.backgroundColor = .clear
        themeDesc.font = .systemFont(ofSize: 12)
        themeDesc.isUserInteractionEnabled = false
        themeContainer.addSubview(themeDesc)
        themeDesc.snp.makeConstraints { maker in
            maker.height.equalTo(24)
            maker.top.equalTo(themeTitle.snp.bottom).offset(-12)
            maker.leading.equalTo(themeColorBlock.snp.trailing).offset(20)
            maker.trailing.equalTo(themeContainer)
        }
        
        themePickedFlag.text = ""
        themePickedFlag.isUserInteractionEnabled = false
        themeContainer.addSubview(themePickedFlag)
        themePickedFlag.snp.makeConstraints { maker in
            maker.width.equalTo(24)
            maker.height.equalTo(24)
            maker.trailing.equalTo(themeContainer)
            maker.centerY.equalTo(themeContainer)
        }
        themePickedFlag.textColor = .black
        
        pickerViews.append(themePickedFlag)
        
        let gesture:UITapGestureRecognizer = UITapGestureRecognizer(target: self,
                                                                    action: #selector(selectColor(sender:)))
        gesture.numberOfTapsRequired = 1
        gesture.name = "\(id)"
        themeContainer.isUserInteractionEnabled = true
        themeContainer.addGestureRecognizer(gesture)
        
        return themeColorBlock
    }
    
    @objc private func selectColor(sender: UIGestureRecognizer){
        for i in 0...2 {
            if (i == Int(sender.name!)) {
                pickerViews[i].text = "✓"
            } else {
                pickerViews[i].text = ""
            }
        }
    }
    
    @objc private func okBtnItemClicked() {
        var id = 0
        for i in 0...2 {
            if (!pickerViews[i].text!.isEmpty) {
                id = i
            }
        }
        switch id {
        case 0: // white bg
            MainViewController.params.backgroundColor = .white
            navigationController?.popToRootViewController(animated: true)
        case 1: // blue bg
            MainViewController.params.backgroundColor = .visaBlue
            navigationController?.popToRootViewController(animated: true)
        default: // custom
            let colorCodeLength = colorCodeTextField.text?.count ?? 0
            if (colorCodeLength != 3 && colorCodeLength != 6) {
                let alert = UIAlertController(title: "Alert",
                                              message: "Please enter a valid HEX color code.",
                                              preferredStyle: .alert)
                alert.addAction(UIAlertAction(title: "OK", style: .default, handler: nil))
                self.present(alert, animated: true, completion: nil)
                return
            }
            let bgColor = UIColor.init(hexString: "#\(colorCodeTextField.text!)")
            let whiteForegroundContrast = UIColor.contrastRatio(between: bgColor,
                                                                and: UIColor.white)
            let blueForegroundContrast = UIColor.contrastRatio(between: bgColor,
                                                               and: UIColor.visaBlue)
            print("SensoryBranding: \(colorCodeTextField.text!) \(whiteForegroundContrast) \(blueForegroundContrast)")
            if (whiteForegroundContrast < 3.0 && blueForegroundContrast < 3.0) {
                let alert = UIAlertController(title: "Alert",
                                              message: "Your custom color doesn't provide enough contrast. Please enter another color.",
                                              preferredStyle: .alert)
                alert.addAction(UIAlertAction(title: "OK", style: .default, handler: nil))
                self.present(alert, animated: true, completion: nil)
            } else {
                MainViewController.params.backgroundColor = bgColor
                navigationController?.popToRootViewController(animated: true)
            }
        }
    }
    
}

extension ColorSelectionViewController: UITextFieldDelegate {
    
    static let maxLength = 6
    
    func textField(_ textField: UITextField,
                   shouldChangeCharactersIn range: NSRange,
                   replacementString string: String) -> Bool {
        let currentString = (textField.text ?? "") as NSString
        let newString = currentString.replacingCharacters(in: range, with: string)
        let lengthIsLessThanOrEqualToMax = newString.count <= ColorSelectionViewController.maxLength
        
        if (newString.count == 3 || newString.count == 6) {
            updateCustomColorBlock(colorInHex: newString)
        }
        
        let disallowedCharacterSet = NSCharacterSet(charactersIn: "0123456789abcdefABCDEF").inverted
        let replacementStringIsLegal = string.rangeOfCharacter(from: disallowedCharacterSet) == nil
        return lengthIsLessThanOrEqualToMax && replacementStringIsLegal
    }
    
    private func updateCustomColorBlock(colorInHex: String) {
        customColorBlock.backgroundColor = UIColor.init(hexString: "#\(colorInHex)")
    }
    
    /**
     * To lose focus when tap out of textfield
     */
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        textField.resignFirstResponder()
        return true
    }
    
    /**
     * To make scroll up views when keyboard shows
     */
    
    func registerForKeyboardNotifications(){
        //Adding notifies on keyboard appearing
        NotificationCenter.default
            .addObserver(self,
                         selector: #selector(keyboardWasShown(notification:)),
                         name: UIResponder.keyboardWillShowNotification,
                         object: nil)
        NotificationCenter.default
            .addObserver(self,
                         selector: #selector(keyboardWillBeHidden(notification:)),
                         name: UIResponder.keyboardWillHideNotification,
                         object: nil)
    }

    func deregisterFromKeyboardNotifications(){
        //Removing notifies on keyboard appearing
        NotificationCenter.default
            .removeObserver(self,
                            name: UIResponder.keyboardWillShowNotification,
                            object: nil)
        NotificationCenter.default
            .removeObserver(self,
                            name: UIResponder.keyboardWillHideNotification,
                            object: nil)
    }

    @objc func keyboardWasShown(notification: NSNotification){
        //Need to calculate keyboard exact size due to Apple suggestions
        self.scrollView.isScrollEnabled = true
        let info = notification.userInfo!
        let keyboardSize = (info[UIResponder.keyboardFrameEndUserInfoKey] as? NSValue)?.cgRectValue.size
        let contentInsets : UIEdgeInsets = UIEdgeInsets(top: 0.0, left: 0.0, bottom: keyboardSize!.height, right: 0.0)

        self.scrollView.contentInset = contentInsets
        self.scrollView.scrollIndicatorInsets = contentInsets

        var aRect : CGRect = self.view.frame
        aRect.size.height -= keyboardSize!.height
        if let activeField = self.activeField {
            if (!aRect.contains(activeField.frame.origin)){
                self.scrollView.scrollRectToVisible(activeField.frame, animated: true)
            }
        }
    }

    @objc func keyboardWillBeHidden(notification: NSNotification){
        //Once keyboard disappears, restore original positions
        let info = notification.userInfo!
        let keyboardSize = (info[UIResponder.keyboardFrameEndUserInfoKey] as? NSValue)?.cgRectValue.size
        let contentInsets : UIEdgeInsets = UIEdgeInsets(top: 0.0, left: 0.0, bottom: -keyboardSize!.height, right: 0.0)
        self.scrollView.contentInset = contentInsets
        self.scrollView.scrollIndicatorInsets = contentInsets
        self.view.endEditing(true)
        self.scrollView.isScrollEnabled = false
    }

    func textFieldDidBeginEditing(_ textField: UITextField){
        pickerViews[0].text = ""
        pickerViews[1].text = ""
        pickerViews[2].text = "✓"
        activeField = textField
    }

    func textFieldDidEndEditing(_ textField: UITextField){
        activeField = nil
    }
    
}
