//
//  MainViewController.swift
//  VisaSensoryBrandingDemo
//

import Foundation
import UIKit
import VisaSensoryBranding

class MainViewController: UIViewController {
    
    let sensoryBranding:SensoryBranding = SensoryBranding.init()
    static var params = AnimationParameters(langCode: "en",
                                     backgroundColor: .white,
                                     enableAudio: false,
                                     enableHaptics: false,
                                     enableCheckmark: false,
                                     checkmarkText: .none)
    
    static let whiteTheme = ThemeDescriptor(title: "Default Light Theme", desc: "White background with full color elements")
    static let blueTheme = ThemeDescriptor(title: "Default Dark Theme", desc: "Visa Blue background with full color elements")
    static let customTheme = ThemeDescriptor(title: "Custom", desc: "With Visa monochrome elements")
    static let customThemeWithHint = ThemeDescriptor(title: "Custom (fill in the color code below)", desc: "With Visa monochrome elements")
    
    var langs = [String: LangDescriptor]()
    
    let containerHeight = 80
    let containerPadding = 20
    let labelSize = 256
    
    
    let animContainer = UIView()
    
    
    
    let scrollView = UIScrollView()
    let stackView = UIStackView()
    
    let themeContainer = UIView()
    let themeColorBlock = UIView()
    let themeTitle = UILabel()
    let themeDesc = UILabel()
    
    let tagLanguage = 0x0000
    let languageContainer = UIView()
    let languageSelectBtn = UIButton()
    
    let audioContainer = UIView()
    let audioSwitch = UISwitch()
    
    let hapticsContainer = UIView()
    let hapticsSwitch = UISwitch()
    
        
    let tagText = 0x0010
    let checkmarkModeContainer = UIView()
    let checkmarkModeSwitch = UISwitch()
    let checkmarkTextContainer = UIView()
    let checkmarkTextSelectBtn = UIButton()
    let checkmarkPicker = UIPickerView()
    
    
    let constrainedContainer = UIView()
    let constrainedSwitch = UISwitch()
    let constrainedWidthContainer = UIView()
    let constrainedWidthTextField = UITextField()
    
    var activeField: UITextField?
    var isPlaying = false
    
    override func viewWillAppear(_ animated: Bool) {
        print("will appear")
        let currColor = MainViewController.params.backgroundColor
        themeColorBlock.backgroundColor = currColor
        switch currColor {
        case .white:
            themeTitle.text = MainViewController.whiteTheme.title
            themeDesc.text = MainViewController.whiteTheme.desc
        
        case .visaBlue:
            themeTitle.text = MainViewController.blueTheme.title
            themeDesc.text = MainViewController.blueTheme.desc
        
        default:
            themeTitle.text = MainViewController.customTheme.title + " - " + currColor.hexString!
            themeDesc.text = MainViewController.customTheme.desc
        }
    }
    
    override func viewDidLoad() {
        view.backgroundColor = .white
        setupLocales()
        
        // Top -> Bottom views
        setupNavBar()
        setupContainers()
        setupThemeOptions()
        setupLanguageOptions()
        setupAudioOptions()
        setupHapticsOptions()
        setupCheckmarkOptions()
        setupSizeConstraintOptions()
        setupAnimContainer()
        
        let tapGesture = UITapGestureRecognizer(target: self, action: #selector(self.dismissKeyboard (_:)))
        self.view.addGestureRecognizer(tapGesture)
    }
    
    func setupLocales() {
        if let url = Bundle(for: SensoryBranding.self)
            .url(forResource: "languages", withExtension: "json") {
            do {
                let languagesData = try Data(contentsOf: url)
                let jsonDecoder = JSONDecoder()
                
                langs = try jsonDecoder.decode([String: LangDescriptor].self,
                                                       from: languagesData)
                let sortedLangs = langs.sorted { $0.value.name < $1.value.name }
                sortedLangs.forEach { (key: String, value: LangDescriptor) in
                    languageDataSource.append(value.name)
                    langCodes.append(value.code)
                }
                
            } catch {
                // Logger.error(error)
            }
        }
    }
    
    
    @objc func dismissKeyboard (_ sender: UITapGestureRecognizer) {
        constrainedWidthTextField.resignFirstResponder()
    }

    
    
    /// Navigation
    private func setupNavBar() {
        let playBarButton = UIBarButtonItem(barButtonSystemItem: .play,
                                            target: self,
                                            action: #selector(play))
        playBarButton.tintColor = .white
        navigationItem.rightBarButtonItem = playBarButton
        
        navigationItem.title = "Visa Sensory Branding"
        
    }
    
    @objc private func play() {
        
        var targetWidth = 0
        if (constrainedWidthTextField.text == nil
            || constrainedWidthTextField.text == ""
            || Int(constrainedWidthTextField.text!) == 0) {
            print("wrong input of width")
            alertOnInvalidWidthInput()
            return
        } else {
            if (constrainedSwitch.isOn) {
                targetWidth = Int(constrainedWidthTextField.text!)!
            } else {
                targetWidth = Int.max
            }
        }
        
        if (!isPlaying) {
            isPlaying = true
                        
            // Config all sensoryBranding fields with params
            sensoryBranding.backdropColor = MainViewController.params.backgroundColor
            sensoryBranding.languageCode = MainViewController.params.langCode
            sensoryBranding.isSoundEnabled = MainViewController.params.enableAudio
            sensoryBranding.isHapticFeedbackEnabled = MainViewController.params.enableHaptics
            print("backgroundColor: " + MainViewController.params.backgroundColor.hexString!)
            print("locale: " + MainViewController.params.langCode)
            print("checkmark: " + String(MainViewController.params.enableCheckmark))
            print("checkmark text: " + String(MainViewController.params.checkmarkText.rawValue))
            if MainViewController.params.enableCheckmark {
                if MainViewController.params.checkmarkText == .none {
                    sensoryBranding.checkmarkMode = .checkmark
                } else if MainViewController.params.checkmarkText == .approve {
                    sensoryBranding.checkmarkMode = .checkmarkWithText
                    sensoryBranding.checkmarkTextOption = .approve
                } else if MainViewController.params.checkmarkText == .success {
                    sensoryBranding.checkmarkMode = .checkmarkWithText
                    sensoryBranding.checkmarkTextOption = .success
                } else {
                    sensoryBranding.checkmarkMode = .checkmarkWithText
                    sensoryBranding.checkmarkTextOption = .complete
                }
            } else {
                sensoryBranding.checkmarkMode = .none
            }
            
            if (constrainedSwitch.isOn) { // Constranit
                sensoryBranding.layer.borderWidth = 1 // .addExternalBorder()
                animContainer.backgroundColor = .white
            } else { // full screen
                self.navigationController?
                    .setNavigationBarHidden(true, animated: false)
                sensoryBranding.layer.borderWidth = 0 // .removeExternalBorders()
                animContainer.backgroundColor = sensoryBranding.backdropColor
            }
            
            sensoryBranding.snp.updateConstraints { make in
                make.width.equalTo(targetWidth)
                make.height.equalTo(Int(Double(targetWidth) * 0.32))
            }
            animContainer.isHidden = false
            
            // Trigger the animation after the size change is done
            
            sensoryBranding.animate { result, err in
                print("sensoryBranding animate result: \(result)")
                self.isPlaying = false
                self.navigationController?
                    .setNavigationBarHidden(false, animated: false)
                self.animContainer.isHidden = true
                if let _ = err {
                    print("Error: \(err!)")
                }
            }
            
        }
    }
    
    private func alertOnInvalidWidthInput() {
        let alert = UIAlertController(title: "Alert",
                                      message: "Please type valid width.",
                                      preferredStyle: .alert)
        alert.addAction(UIAlertAction(title: "OK", style: .default, handler: { action in
            switch action.style{
                case .default:
                print("default")
                
                case .cancel:
                print("cancel")
                
                case .destructive:
                print("destructive")
                
            @unknown default:
                fatalError()
            }
        }))
        self.present(alert, animated: true, completion: nil)
    }
    
    /// Theme
    private func setupThemeOptions() {
        themeColorBlock.backgroundColor = .white
        themeColorBlock.layer.borderWidth = 1
        themeColorBlock.layer.borderColor = UIColor.darkGray.cgColor
        themeContainer.addSubview(themeColorBlock)
        themeColorBlock.snp.makeConstraints { maker in
            maker.width.equalTo(40)
            maker.height.equalTo(40)
            maker.leading.equalTo(themeContainer)
            maker.centerY.equalTo(themeContainer.snp.centerY)
        }
        
        themeTitle.text = "Default Light Theme"
        themeTitle.textColor = .black
        themeTitle.font = .systemFont(ofSize: 14)
        themeTitle.isUserInteractionEnabled = false
        themeContainer.addSubview(themeTitle)
        themeTitle.snp.makeConstraints { maker in
            maker.height.equalTo(32)
            maker.top.equalTo(themeColorBlock.snp.top).offset(-4)
            maker.leading.equalTo(themeColorBlock.snp.trailing).offset(20)
            maker.trailing.equalTo(themeContainer)
        }
        
        
        themeDesc.text = "White background with full color elements"
        themeDesc.textColor = .lightGray
        themeDesc.backgroundColor = .clear
        themeDesc.font = .systemFont(ofSize: 12)
        themeDesc.isUserInteractionEnabled = false
        themeContainer.addSubview(themeDesc)
        themeDesc.snp.makeConstraints { maker in
            maker.height.equalTo(24)
            maker.top.equalTo(themeTitle.snp.bottom).offset(-12)
            maker.leading.equalTo(themeColorBlock.snp.trailing).offset(20)
            maker.trailing.equalTo(themeContainer)
        }
        
        
        let gesture:UITapGestureRecognizer = UITapGestureRecognizer(target: self, action: #selector(navigateToColorSelectionVC))
        gesture.numberOfTapsRequired = 1
        themeContainer.isUserInteractionEnabled = true
        themeContainer.addGestureRecognizer(gesture)
    }
    
    @objc private func navigateToColorSelectionVC(){
        navigationController?.pushViewController(
            ColorSelectionViewController(),
            animated: true)
    }
    
    
    
    private func setupLanguageOptions() {
        createOptionLabel(container: languageContainer, labelTitle: "Language")
        
        languageSelectBtn.setTitle("English", for: .normal)
        languageSelectBtn.setTitleColor(.visaBlue, for: .normal)
        languageSelectBtn.contentHorizontalAlignment = .trailing
        languageSelectBtn.titleLabel?.font = UIFont.systemFont(ofSize: 12)
        languageSelectBtn.tag = tagLanguage
        languageContainer.addSubview(languageSelectBtn)
        languageSelectBtn.snp.makeConstraints { make in
            make.width.equalTo(180)
            make.trailing.equalTo(languageContainer)
            make.centerY.equalTo(languageContainer)
        }
        
        (languageSelectBtn).addTarget(self, action: #selector(popUpPicker(sender:)), for: .touchUpInside)
    }
    
    
    /// Audio
    private func setupAudioOptions() {
        createOptionLabel(container: audioContainer, labelTitle: "Audio")
        
        audioSwitch.setOn(false, animated: false)
        audioSwitch.onTintColor = .visaBlue
        audioContainer.addSubview(audioSwitch)
        audioSwitch.alignTraillingCenterYOfParent(parent: audioContainer)
        
        audioSwitch.addTarget(self, action: #selector(enableAudio(sender:)), for: .touchUpInside)
    }
    
    @objc func enableAudio(sender: UISwitch) {
        MainViewController.params.enableAudio = sender.isOn
    }
    
    
    /// Haptics
    private func setupHapticsOptions() {
        createOptionLabel(container: hapticsContainer, labelTitle: "Haptic")
        
        hapticsSwitch.setOn(false, animated: false)
        hapticsSwitch.onTintColor = .visaBlue
        hapticsContainer.addSubview(hapticsSwitch)
        hapticsSwitch.alignTraillingCenterYOfParent(parent: hapticsContainer)
        
        hapticsSwitch.addTarget(self, action: #selector(enableHaptics(sender:)), for: .touchUpInside)
    }
    
    @objc func enableHaptics(sender: UISwitch) {
        MainViewController.params.enableHaptics = sender.isOn
    }
    
    
    /// Checkmark Language & Text
    private func setupCheckmarkOptions() {
        
        // Mode
        createOptionLabel(container: checkmarkModeContainer, labelTitle: "Checkmark")
        
        checkmarkModeSwitch.setOn(false, animated: false)
        checkmarkModeSwitch.onTintColor = .visaBlue
        
        checkmarkModeContainer.addSubview(checkmarkModeSwitch)
        checkmarkModeSwitch.alignTraillingCenterYOfParent(parent: checkmarkModeContainer)
        
        checkmarkModeSwitch.addTarget(self, action: #selector(enabledCheckMark(sender:)), for: .touchUpInside)
        
        
        
        // Text
        createOptionLabel(container: checkmarkTextContainer, labelTitle: "Checkmark Text")
        
        
        checkmarkTextSelectBtn.setTitle("None", for: .normal)
        checkmarkTextSelectBtn.setTitleColor(.visaBlue, for: .normal)
        checkmarkTextSelectBtn.contentHorizontalAlignment = .trailing
        checkmarkTextSelectBtn.tag = tagText
        checkmarkTextContainer.addSubview(checkmarkTextSelectBtn)
        checkmarkTextSelectBtn.snp.makeConstraints { make in
            make.width.equalTo(120)
            make.trailing.equalTo(checkmarkTextContainer)
            make.centerY.equalTo(checkmarkTextContainer)
        }
        
        // Bottomsheet popup for checkmark text pick-up
        checkmarkTextSelectBtn.tag = tagText
        checkmarkTextSelectBtn.titleLabel?.font = .systemFont(ofSize: 12)
        checkmarkTextSelectBtn.addTarget(self, action: #selector(popUpPicker), for: .touchUpInside)
        checkmarkTextContainer.isHidden = true
    }
    
    @objc func enabledCheckMark(sender: UISwitch){
        let enabled = sender.isOn
        self.checkmarkTextContainer.isHidden = !enabled
        MainViewController.params.enableCheckmark = enabled
    }
    
    @objc func popUpPicker(sender: UIButton){
        let screenWidth = UIScreen.main.bounds.width - 20
        let screenHeight = UIScreen.main.bounds.height / 3
        let vc = UIViewController()
        vc.preferredContentSize = CGSize(width: screenWidth, height: screenHeight)
        let pickerView = UIPickerView(frame: CGRect(x: 0, y: 0, width: screenWidth, height:screenHeight))
        pickerView.tag = sender.tag
        pickerView.dataSource = self
        pickerView.delegate = self
        pickerView.selectRow(0, inComponent: 0, animated: false)
            
        vc.view.addSubview(pickerView)
        pickerView.snp.makeConstraints { make in
            make.centerX.equalTo(vc.view)
            make.centerY.equalTo(vc.view)
        }
        
        var title = ""
        if (sender.tag == tagText) {
            title = "Select Checkmark Text"
        } else {
            title = "Select Language"
        }
        let alert = UIAlertController(title: title,
                                      message: "",
                                      preferredStyle: .actionSheet)
        alert.popoverPresentationController?.sourceView = checkmarkTextSelectBtn
        alert.popoverPresentationController?.sourceRect = checkmarkTextSelectBtn.bounds
        alert.setValue(vc, forKey: "contentViewController")
        alert.addAction(UIAlertAction(title: "Cancel", style: .cancel, handler: { (UIAlertAction) in
        }))
            
        alert.addAction(UIAlertAction(title: "Select", style: .default, handler: { (UIAlertAction) in
            let selectedRow = pickerView.selectedRow(inComponent: 0)
            if (pickerView.tag == self.tagText) {
                
                let selectedText = checkmarkTextDataSource[selectedRow]
                print("SensoryBranding: checkmarkText selectedRow \(selectedRow)")
                MainViewController.params.checkmarkText = CheckmarkTextUIOptions(rawValue: selectedRow) ?? CheckmarkTextUIOptions.none
                self.checkmarkTextSelectBtn.setTitle(selectedText, for: .normal)
            } else {
                let text = languageDataSource[selectedRow]
                let code = langCodes[selectedRow]
                MainViewController.params.langCode = code
                self.languageSelectBtn.setTitle(text, for: .normal)
                
                checkmarkTextDataSource.removeAll()
                let lang = self.langs[code]
                checkmarkTextDataSource.append("None")
                checkmarkTextDataSource.append(lang!.data.approve)
                checkmarkTextDataSource.append(lang!.data.success)
                checkmarkTextDataSource.append(lang!.data.complete)
                                
                MainViewController.params.checkmarkText = .none
                self.checkmarkTextSelectBtn.setTitle("None", for: .normal)
            }
        }))
            
        self.present(alert, animated: true, completion: nil)
    }

    
    
    /// Size constraint rules
    private func setupSizeConstraintOptions() {
        createOptionLabel(container: constrainedContainer, labelTitle: "Constrained View")
        
        constrainedSwitch.setOn(false, animated: false)
        constrainedSwitch.onTintColor = .visaBlue
        constrainedContainer.addSubview(constrainedSwitch)
        constrainedSwitch.snp.makeConstraints { make in
            make.trailing.equalTo(constrainedContainer)
            make.centerY.equalTo(constrainedContainer)
        }
        constrainedSwitch.addTarget(self, action: #selector(enableConstraintViews(sender:)), for: .touchUpInside)
        
        createOptionLabel(container: constrainedWidthContainer, labelTitle: "Constrained Width (pt)")
        constrainedWidthContainer.addSubview(constrainedWidthTextField)
        constrainedWidthTextField.alignTraillingCenterYOfParent(parent: constrainedWidthContainer)
        constrainedWidthTextField.textAlignment = .right
        constrainedWidthTextField.layer.borderWidth = 1.0
        constrainedWidthTextField.snp.makeConstraints { make in
            make.width.equalTo(120)
        }
        constrainedWidthTextField.addTarget(self, action: #selector(constraintWidthDidChange(_:)), for: .editingChanged)
        constrainedWidthTextField.delegate = self
        
        constrainedWidthTextField.keyboardType = .numberPad
        constrainedWidthTextField.textColor = .black
        constrainedWidthTextField.text = "200"
        constrainedWidthContainer.isHidden = true
    }
    
    
    @objc func enableConstraintViews(sender: UISwitch) {
        let enabled = sender.isOn
        constrainedWidthContainer.isHidden = !enabled

        if enabled {
            registerForKeyboardNotifications()
        } else {
            deregisterFromKeyboardNotifications()
        }
    }
    
    @objc func constraintWidthDidChange(_ textField: UITextField) {

    }
    
    @objc func constraintHeightDidChange(_ textField: UITextField) {

    }
    
    
    private func setupAnimContainer() {
        self.view.addSubview(animContainer)
        animContainer.snp.makeConstraints { make in
            make.edges.equalTo(self.view)
        }
        animContainer.backgroundColor = .white
        animContainer.isHidden = true
        
        animContainer.addSubview(sensoryBranding)
        sensoryBranding.layer.borderColor = UIColor.darkGray.cgColor
        sensoryBranding.snp.makeConstraints { make in
            make.centerX.equalTo(self.animContainer)
            make.centerY.equalTo(self.animContainer)
            let targetWidth = 200
            make.width.equalTo(targetWidth)
            make.height.equalTo(Int(Double(targetWidth) * 0.32))
        }
    
    }
    
    
    private func setupContainers() {
        view.addSubview(scrollView)
        
        scrollView.snp.makeConstraints { make in
            make.edges.equalTo(view.safeAreaLayoutGuide)
        }
        scrollView.addSubview(stackView)
        
        stackView.axis = .vertical
        stackView.alignment = .fill
        stackView.distribution = .equalSpacing
        stackView.spacing = 0
        stackView.snp.makeConstraints { make in
            make.width.equalTo(scrollView.snp.width).offset(containerPadding * -2)
            make.leading.equalTo(scrollView.snp.leading).offset(containerPadding)
            make.trailing.equalTo(scrollView.snp.trailing)
            make.top.equalTo(scrollView.snp.top)
            make.bottom.equalTo(scrollView.snp.bottom)
        }
        
        let containerDividerPairs: KeyValuePairs = [themeContainer: true,
         languageContainer: true,
         audioContainer: true,
         hapticsContainer: true,
         checkmarkModeContainer: false,
         checkmarkTextContainer: true,
         constrainedContainer: false,
         constrainedWidthContainer: true]
        
        containerDividerPairs.forEach { container, enableDivider in
            stackView.addArrangedSubview(container)
            container.snp.makeConstraints { maker in
                maker.height.equalTo(containerHeight)
                maker.leading.equalTo(stackView.snp.leading)
                maker.trailing.equalTo(stackView.snp.trailing)
            }
            if (enableDivider) {
                let divider = UIView()
                divider.backgroundColor = .dividerGray
                stackView.addArrangedSubview(divider)
                divider.snp.makeConstraints { maker in
                    maker.height.equalTo(1)
                    maker.leading.equalTo(stackView.snp.leading)//.offset(containerPadding)
                    maker.trailing.equalTo(stackView.snp.trailing)//.offset(containerPadding * -1)
                }
            }
        }
    
    }
    
    private func createOptionLabel(container: UIView, labelTitle: String) {
        let label = UILabel()
        label.text = labelTitle
        label.textColor = .black
        label.font = .systemFont(ofSize: 14)
        container.addSubview(label)
        label.snp.makeConstraints { make in
            make.width.equalTo(labelSize)
            make.leading.equalTo(container)
            make.centerY.equalTo(container)
        }
    }
    
}

extension UIView {
    
    func alignTraillingCenterYOfParent(parent: UIView) {
        self.snp.makeConstraints { make in
            make.trailing.equalTo(parent)
            make.centerY.equalTo(parent)
        }
    }
    
    struct Constants {
        static let ExtenalBorderLayerName = "externalBorder"
    }
    
    func addExternalBorder(borderWidth: CGFloat = 1.0,
                           borderColor: UIColor = .black) -> CALayer {
        let externalBorder = CALayer()
        externalBorder.frame = CGRect(x: -borderWidth,
                                      y: -borderWidth,
                                      width: frame.size.width + 2 * borderWidth,
                                      height: frame.size.height + 2 * borderWidth)
        externalBorder.borderColor = borderColor.cgColor
        externalBorder.borderWidth = borderWidth
        externalBorder.name = Constants.ExtenalBorderLayerName
        
        layer.insertSublayer(externalBorder, at: 0)
        layer.masksToBounds = false
        
        return externalBorder
    }
    
    func removeExternalBorders() {
        layer.sublayers?.filter{
            $0.name == Constants.ExtenalBorderLayerName
        }.forEach() {
            $0.removeFromSuperlayer()
        }
    }
    
    func removeExternalBorder(externalBorder: CALayer) {
        guard externalBorder.name == Constants.ExtenalBorderLayerName
                else { return }
        externalBorder.removeFromSuperlayer()
    }
}

extension MainViewController: UITextFieldDelegate {

    /**
     * To lose focus when tap out of textfield
     */
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        textField.resignFirstResponder()
        return true
    }
    
    /**
     * To make scroll up views when keyboard shows
     */
    
    func registerForKeyboardNotifications(){
        //Adding notifies on keyboard appearing
        NotificationCenter.default
            .addObserver(self,
                         selector: #selector(keyboardWasShown(notification:)),
                         name: UIResponder.keyboardWillShowNotification,
                         object: nil)
        NotificationCenter.default
            .addObserver(self,
                         selector: #selector(keyboardWillBeHidden(notification:)),
                         name: UIResponder.keyboardWillHideNotification,
                         object: nil)
    }

    func deregisterFromKeyboardNotifications(){
        //Removing notifies on keyboard appearing
        NotificationCenter.default
            .removeObserver(self,
                            name: UIResponder.keyboardWillShowNotification,
                            object: nil)
        NotificationCenter.default
            .removeObserver(self,
                            name: UIResponder.keyboardWillHideNotification,
                            object: nil)
    }

    @objc func keyboardWasShown(notification: NSNotification){
        //Need to calculate keyboard exact size due to Apple suggestions
        let userInfo: NSDictionary = notification.userInfo! as NSDictionary
        let keyboardInfo = userInfo[UIResponder.keyboardFrameBeginUserInfoKey] as! NSValue
        let keyboardSize = keyboardInfo.cgRectValue.size
        let contentInsets = UIEdgeInsets(top: 0, left: 0, bottom: keyboardSize.height, right: 0)
        scrollView.contentInset = contentInsets
        scrollView.scrollIndicatorInsets = contentInsets
    }

    @objc func keyboardWillBeHidden(notification: NSNotification){
        //Once keyboard disappears, restore original positions
        scrollView.contentInset = .zero
        scrollView.scrollIndicatorInsets = .zero
    }

    func textFieldDidBeginEditing(_ textField: UITextField){
        activeField = textField
    }

    func textFieldDidEndEditing(_ textField: UITextField){
        activeField = nil
    }
    
}

struct ThemeDescriptor {
    public var title: String
    public var desc: String
}
